import {
  Directive,
  Input,
  Output,
} from "@angular/core";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {ApplicationRolePermissionEnum} from "@app/shared/enums/application-role-permission.enum";
import {
  AppRoutesEnum,
  OrderRoutesEnum,
  RoutesEnum,
} from "@app/shared/enums/routes.enum";
import {ThemeEnum} from "@app/shared/enums/theme.enum";
import {PermissionUtil} from "@app/shared/utils/permission.util";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {User} from "@models";
import {DataTestEnum} from "@shared/enums/data-test.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  OAuth2Service,
  ObservableUtil,
  urlSeparator,
} from "solidify-frontend";

@Directive()
export abstract class AbstractMainToolbarPresentational extends SharedAbstractPresentational {
  @Input()
  logged: boolean = false;

  @Input()
  currentPath: string;

  @Input()
  currentLanguage: Enums.Language.LanguageEnum;

  @Input()
  user: User;

  @Input()
  photoUser: string;

  institutionUrl: string = environment.institutionUrl;

  @Input()
  numberArchiveInCart: number = 0;

  @Input()
  numberMyOrderReadyNew: number = 0;

  @Input()
  numberPendingRequestNotificationInbox: number = 0;

  @Input()
  private userRoles: Enums.UserApplicationRole.UserApplicationRoleEnum[];

  // Theme
  private readonly _themeBS: BehaviorSubject<ThemeEnum | undefined> = new BehaviorSubject<ThemeEnum | undefined>(undefined);

  @Output("themeChange")
  readonly themeObs: Observable<ThemeEnum | undefined> = ObservableUtil.asObservable(this._themeBS);

  isProduction: boolean = environment.production;
  isDemoMode: boolean = environment.isDemoMode;
  isOpen: boolean = false;

  private get _theme(): ThemeEnum | undefined {
    return this._themeBS.getValue();
  }

  private set _theme(value: ThemeEnum | undefined) {
    this._themeBS.next(value);
  }

  get theme(): ThemeEnum | undefined {
    return this._theme;
  }

  @Input()
  set theme(theme: ThemeEnum | undefined) {
    this._theme = theme;
  }

  // Navigate
  protected readonly _navigateBS: BehaviorSubject<string | undefined> = new BehaviorSubject<string | undefined>(undefined);
  @Output("navigateChange")
  readonly navigateObs: Observable<string | undefined> = ObservableUtil.asObservable(this._navigateBS);

  // Language
  private readonly _languageBS: BehaviorSubject<Enums.Language.LanguageEnum | undefined> = new BehaviorSubject<Enums.Language.LanguageEnum | undefined>(undefined);
  @Output("languageChange")
  readonly languageObs: Observable<Enums.Language.LanguageEnum | undefined> = ObservableUtil.asObservable(this._languageBS);

  // Logout
  private readonly _logoutBS: BehaviorSubject<void | undefined> = new BehaviorSubject<void | undefined>(undefined);
  @Output("logoutChange")
  readonly logoutObs: Observable<void | undefined> = ObservableUtil.asObservable(this._logoutBS);

  private listMenus: MenuToolbar[] = [
    {
      click: (menu) => this.navigate(menu.path()),
      path: () => RoutesEnum.homePage,
      rootModulePath: RoutesEnum.homePage,
      labelToTranslate: LabelTranslateEnum.home,
      isVisible: () => true,
      icon: IconNameEnum.home,
      dataTest: DataTestEnum.linkMenuHome,
    },
    {
      click: (menu) => this.openOrder(),
      path: () => RoutesEnum.order,
      rootModulePath: RoutesEnum.order,
      labelToTranslate: LabelTranslateEnum.archiveOrders,
      isVisible: () => PermissionUtil.isUserHavePermission(this.logged, ApplicationRolePermissionEnum.userPermission, this.userRoles),
      icon: IconNameEnum.order,
      badgeCounter: () => "" + this.numberMyOrderReadyNew,
      badgeDescription: "Counter of new order ready in my order",
      badgeHidden: () => this.numberMyOrderReadyNew === 0,
      dataTest: DataTestEnum.linkMenuOrder,
      tourAnchor: TourEnum.mainMenuOrder,
    },
    {
      click: (menu) => this.navigate(menu.path()),
      path: () => RoutesEnum.deposit,
      rootModulePath: RoutesEnum.deposit,
      labelToTranslate: LabelTranslateEnum.deposit,
      isVisible: () => PermissionUtil.isUserHavePermission(this.logged, ApplicationRolePermissionEnum.userPermission, this.userRoles),
      icon: IconNameEnum.deposit,
      dataTest: DataTestEnum.linkMenuDeposit,
      tourAnchor: TourEnum.mainMenuDeposit,
    },
    {
      click: (menu) => this.navigate(menu.path()),
      path: () => this.numberPendingRequestNotificationInbox === 0 ? RoutesEnum.preservationSpace : RoutesEnum.preservationSpaceNotificationInbox,
      rootModulePath: RoutesEnum.preservationSpace,
      labelToTranslate: LabelTranslateEnum.preservationSpace,
      isVisible: () => PermissionUtil.isUserHavePermission(this.logged, ApplicationRolePermissionEnum.userPermission, this.userRoles),
      icon: IconNameEnum.preservationSpace,
      badgeCounter: () => "" + this.numberPendingRequestNotificationInbox,
      badgeDescription: "Counter of pending request notification in inbox",
      badgeHidden: () => this.numberPendingRequestNotificationInbox === 0,
      dataTest: DataTestEnum.linkMenuPreservationSpace,
      tourAnchor: TourEnum.mainMenuPreservationSpace,
    },
  ];

  private listMenusAdmin: MenuToolbar[] = [
    {
      click: (menu) => this.navigate(menu.path()),
      path: () => RoutesEnum.preservationPlanning,
      rootModulePath: RoutesEnum.preservationPlanning,
      labelToTranslate: LabelTranslateEnum.preservationPlanning,
      isVisible: () => PermissionUtil.isUserHavePermission(this.logged, ApplicationRolePermissionEnum.adminPermission, this.userRoles),
      icon: IconNameEnum.preservationPlanning,
      dataTest: DataTestEnum.linkMenuPreservationPlanning,
    },
    {
      click: (menu) => this.navigate(menu.path()),
      path: () => RoutesEnum.admin,
      rootModulePath: RoutesEnum.admin,
      labelToTranslate: LabelTranslateEnum.administration,
      isVisible: () => PermissionUtil.isUserHavePermission(this.logged, ApplicationRolePermissionEnum.adminPermission, this.userRoles),
      icon: IconNameEnum.administration,
      dataTest: DataTestEnum.linkMenuAdmin,
    },
  ];

  get routesEnum(): typeof RoutesEnum {
    return RoutesEnum;
  }

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  protected constructor(private oauthService: OAuth2Service) {
    super();
  }

  getListMenuUser(): MenuToolbar[] {
    return this.listMenus.filter(m => m.isVisible());
  }

  getListMenuAdmin(): MenuToolbar[] {
    return this.listMenusAdmin.filter(m => m.isVisible());
  }

  getListMenuUserAndAdminMenu(): MenuToolbar[] {
    return [...this.getListMenuUser(), ...this.getListMenuAdmin()];
  }

  navigate(path: string): void {
    this._navigateBS.next(path);
  }

  useLanguage(language: Enums.Language.LanguageEnum): void {
    this._languageBS.next(language);
  }

  toggleMenu(): void {
    this.isOpen = !this.isOpen;
  }

  openCart(): void {
    this.navigate(RoutesEnum.orderMyOrderDraft);
  }

  openOrder(): void {
    if (this.userRoles.includes(Enums.UserApplicationRole.UserApplicationRoleEnum.admin) || this.userRoles.includes(Enums.UserApplicationRole.UserApplicationRoleEnum.root)) {
      this.navigate(AppRoutesEnum.order);
    } else {
      this.navigate(AppRoutesEnum.order + urlSeparator + OrderRoutesEnum.myOrder);
    }
  }

  login(): void {
    this.oauthService.initAuthorizationCodeFlow();
  }

  logout(): void {
    this._logoutBS.next();
  }

  navigateToInstitution(): void {
    window.open(this.institutionUrl, "_blank");
  }
}

export interface MenuToolbar {
  path?: () => string;
  rootModulePath: string;
  click: (menu: MenuToolbar) => void;
  labelToTranslate: string;
  icon: IconNameEnum;
  isVisible: () => boolean;
  badgeCounter?: (() => string) | undefined;
  badgeDescription?: string | undefined;
  badgeHidden?: (() => boolean) | undefined;
  dataTest?: DataTestEnum;
  tourAnchor?: TourEnum | undefined;
}
