import {DOCUMENT} from "@angular/common";
import {
  AfterViewInit,
  Component,
  ElementRef,
  HostListener,
  Inject,
  OnInit,
  Renderer2,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {
  Router,
  RouterStateSnapshot,
} from "@angular/router";
import {UpdateVersionDialog} from "@app/components/dialogs/update-version/update-version.dialog";
import {UserDialog} from "@app/components/dialogs/user/user.dialog";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {
  RoutesEnum,
  urlSeparator,
} from "@app/shared/enums/routes.enum";
import {ThemeEnum} from "@app/shared/enums/theme.enum";
import {LocalStateModel} from "@app/shared/models/local-state.model";
import {AppAction} from "@app/stores/app.action";
import {AppState} from "@app/stores/app.state";
import {AppBannerState} from "@app/stores/banner/app-banner.state";
import {AppCartArchiveState} from "@app/stores/cart/archive/app-cart-archive.state";
import {AppUserState} from "@app/stores/user/app-user.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {HomeHelper} from "@home/helpers/home.helper";
import {User} from "@models";
import {TranslateService} from "@ngx-translate/core";
import {Navigate} from "@ngxs/router-plugin";
import {
  Select,
  Store,
} from "@ngxs/store";
import {ApplicationRolePermissionEnum} from "@shared/enums/application-role-permission.enum";
import {BannerColorEnum} from "@shared/enums/banner-color.enum";
import {ChemicalMoleculeVisualizationEnum} from "@shared/enums/chemical-molecule-visualization.enum";
import {SessionStorageEnum} from "@shared/enums/session-storage.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {SessionStorageHelper} from "@shared/helpers/session-storage.helper";
import {UrlQueryParamHelper} from "@shared/helpers/url-query-param.helper";
import {AppStatusService} from "@shared/services/app-status.service";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {GoogleAnalyticsService} from "@shared/services/google-analytics.service";
import {ScrollService} from "@shared/services/scroll.service";
import {PermissionUtil} from "@shared/utils/permission.util";
import {TourService} from "ngx-tour-md-menu";
import {
  combineLatest,
  fromEvent,
  Observable,
} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  map,
  take,
  tap,
} from "rxjs/operators";
import {
  isFalse,
  isFalsy,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isTrue,
  isUndefined,
  LoggingService,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
} from "solidify-frontend";

declare let gtag: Function;

@Component({
  selector: "dlcm-root",
  templateUrl: "./app.component.html",
  styleUrls: ["./app.component.scss"],
})
export class AppComponent extends SharedAbstractPresentational implements OnInit, AfterViewInit {
  private readonly _ATTRIBUTE_SRC_DIFFER_KEY: string = "src-differ";
  private readonly _ATTRIBUTE_SRC: string = "src";

  private static readonly _themeAttributeName: string = "theme";
  private static readonly _uiAttributeName: string = "ui";
  currentModule: string;
  isHomePage: boolean;
  logo: string;

  @Select((state: LocalStateModel) => state.router.state) urlStateObs: Observable<RouterStateSnapshot>;
  appLanguageObs: Observable<Enums.Language.LanguageEnum> = MemoizedUtil.select(this._store, AppState, state => state.appLanguage);
  isLoggedObs: Observable<boolean> = MemoizedUtil.select(this._store, AppState, state => state.isLoggedIn);
  isApplicationInitializedObs: Observable<boolean> = MemoizedUtil.select(this._store, AppState, state => state.isApplicationInitialized);
  isServerOfflineObs: Observable<boolean> = MemoizedUtil.select(this._store, AppState, state => state.isServerOffline);
  themeObs: Observable<ThemeEnum> = MemoizedUtil.select(this._store, AppState, state => state.theme);
  displayCookieConsentObs: Observable<boolean> = MemoizedUtil.select(this._store, AppState, state => state.displayCookieConsent);
  userRolesObs: Observable<Enums.UserApplicationRole.UserApplicationRoleEnum[]> = MemoizedUtil.select(this._store, AppState, state => state.userRoles);
  currentUserObs: Observable<User> = MemoizedUtil.current(this._store, AppUserState);
  currentPhotoUserObs: Observable<string> = MemoizedUtil.select(this._store, AppUserState, state => state.logo);

  displayBannerObs: Observable<boolean> = MemoizedUtil.select(this._store, AppBannerState, state => state.display);
  messageBannerObs: Observable<string> = MemoizedUtil.select(this._store, AppBannerState, state => state.message);
  colorBannerObs: Observable<BannerColorEnum> = MemoizedUtil.select(this._store, AppBannerState, state => state.color);

  isTouchInterfaceObs: Observable<boolean> = MemoizedUtil.select(this._store, AppState, state => state.isTouchInterface);
  numberArchiveInCartObs: Observable<number> = MemoizedUtil.total(this._store, AppCartArchiveState);

  ignoreGrid: boolean = false;
  displayCart: boolean = false;
  isMaintenanceModeActive: boolean = false;

  @ViewChild("main")
  mainElementRef: ElementRef;

  @HostListener("window:beforeunload", ["$event"])
  leaveWithoutSaveNotification($event: Event): void {
    const preventExit = this._store.selectSnapshot((state: LocalStateModel) => state.application.preventExit);
    const ignorePreventExit = this._store.selectSnapshot((state: LocalStateModel) => state.application.ignorePreventLeavePopup);
    if (isFalse(ignorePreventExit) && isTrue(preventExit)) {
      $event.returnValue = true;
    }
  }

  get sessionStorageHelper(): typeof SessionStorageHelper {
    return SessionStorageHelper;
  }

  get sessionStorageEnum(): typeof SessionStorageEnum {
    return SessionStorageEnum;
  }

  get routesEnum(): typeof RoutesEnum {
    return RoutesEnum;
  }

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  constructor(private readonly _store: Store,
              private readonly _router: Router,
              private readonly _translate: TranslateService,
              private readonly _renderer: Renderer2,
              @Inject(DOCUMENT) private readonly _document: Document,
              private readonly _notificationService: NotificationService,
              private readonly _loggingService: LoggingService,
              public readonly appStatusService: AppStatusService,
              public readonly breakpointService: BreakpointService,
              private readonly _dialog: MatDialog,
              public readonly tourService: TourService,
              private _scrollService: ScrollService,
              private router: Router,
              private readonly _googleAnalyticsService: GoogleAnalyticsService) {
    super();
    this._store.dispatch(new AppAction.InitApplication());
    this._setOverrideCssStyleSheet();
    this._listenCurrentModule();
    this.subscribe(this._observeThemeChange());
    this.observeOfflineOnlineMode();
    this.themeChange(environment.theme);
    this.subscribe(this.observeUpdateAvailableForModal());
    this.subscribe(this.observeLoggedUserForCart());
    this.checkOrcidQueryParam();
    this._activeChemicalMoleculePreviewIfEnable();
    this.subscribe(this.listenLanguageChange());
    this.subscribe(this.listenAppReady());
    this._googleAnalyticsService.init(this._renderer);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.subscribe(this.isApplicationInitializedObs.pipe(
      filter(isInitialized => isTrue(isInitialized)),
      take(1),
      tap(() => this.isMaintenanceModeActive = environment.maintenanceMode),
    ));

    this.subscribe(this.isTouchInterfaceObs, isTouchInterface => this.setTouchInterface(isTouchInterface));
    this.subscribe(this.observeTouchEvent());
  }

  ngAfterViewInit(): void {
    super.ngAfterViewInit();
    this._scrollService.init(this.mainElementRef);
  }

  private observeTouchEvent(): Observable<Event> {
    return fromEvent(document, "touchstart")
      .pipe(
        take(1),
        tap(event => this._store.dispatch(new AppAction.SetIsTouchInterface(true)),
        ),
      );
  }

  private observeOfflineOnlineMode(): void {
    let previousStateIsOnline: boolean | undefined = undefined;
    this.subscribe(this.appStatusService.online.pipe(
      distinctUntilChanged(),
      tap((isOnline: boolean) => {
        if ((isUndefined(previousStateIsOnline) || isTrue(previousStateIsOnline)) && isFalsy(isOnline)) {
          this._notificationService.showWarning(MARK_AS_TRANSLATABLE("app.status.offline"));
          previousStateIsOnline = isOnline;
          return;
        }

        if (isFalse(previousStateIsOnline) && isTrue(isOnline)) {
          this._notificationService.showInformation(MARK_AS_TRANSLATABLE("app.status.backToOnline"));
          previousStateIsOnline = isOnline;
          return;
        }
        return;
      }),
    ));
  }

  private observeUpdateAvailableForModal(): Observable<string> {
    return this.appStatusService.updateAvailable.pipe(
      distinctUntilChanged(),
      filter(version => !isNullOrUndefined(version)),
      tap((version) => {
        this._dialog.open(UpdateVersionDialog);
      }),
    );
  }

  private observeLoggedUserForCart(): Observable<boolean> {
    return combineLatest([this.isLoggedObs, this.userRolesObs])
      .pipe(
        map(([isLogged, userRoles]) => this.computeDisplayCart(isLogged, userRoles)),
      );
  }

  private checkOrcidQueryParam(): void {
    if (UrlQueryParamHelper.contains(environment.orcidQueryParam)) {
      this.subscribe(this.currentUserObs.pipe(
        filter(user => isNotNullNorUndefined(user)),
        take(1),
        tap(user => {
          this._dialog.open(UserDialog, {
            data: user,
            width: "90%",
          });
          this._notificationService.showSuccess(MARK_AS_TRANSLATABLE("notification.orcid.associated"), {orcid: user?.person?.orcid});
        }),
      ));
    }
  }

  useLanguage(language: Enums.Language.LanguageEnum): void {
    this._store.dispatch(new AppAction.ChangeAppLanguage(language));
  }

  navigate(route: string): void {
    this._store.dispatch(new Navigate([route]));
  }

  private _setTitle(): void {
    setTimeout(() => {
      this._document.title = environment.appTitle;
      const description = environment.appDescription;
      this._document.querySelector("meta[name=\"description\"]").setAttribute("content", description);
    }, 0);
  }

  private setTheme(): void {
    this._renderer.setAttribute(this._document.body, AppComponent._themeAttributeName, environment.theme);
  }

  private setTouchInterface(isTouchInterface: boolean): void {
    this._renderer.setAttribute(this._document.body, AppComponent._uiAttributeName, isTouchInterface ? "touch" : "standard");
  }

  private setFavicon(): void {
    const header = this._document.querySelector("head");
    const oldFavicon = this._document.querySelector("head link[type='image/x-icon']");
    let link = this._document.createElement("link");
    if (isNotNullNorUndefined(oldFavicon)) {
      link = oldFavicon as any;
    }
    this._renderer.setAttribute(link, "type", "image/x-icon");
    this._renderer.setAttribute(link, "rel", "shortcut icon");
    this._renderer.setAttribute(link, "href", `./assets/themes/${environment.theme}/favicon.ico`);
    this._renderer.appendChild(header, link);
  }

  private setImageToolbar(): void {
    this.logo = `assets/themes/${environment.theme}/toolbar-header-image.svg`;
  }

  private _listenCurrentModule(): void {
    this.subscribe(this.urlStateObs
      .pipe(
        tap(urlState => {
          this.ignoreGrid = false;
          this.isHomePage = false;
          if (urlState) {
            const url = urlState.url;

            const urlHomeDetail = urlSeparator + RoutesEnum.homeDetail;
            if (url.startsWith(urlHomeDetail)) {
              this.currentModule = undefined;
              if (url.lastIndexOf(urlSeparator) === urlHomeDetail.length) {
                this.ignoreGrid = true;
              }
              return;
            }

            if (url.startsWith(urlSeparator + RoutesEnum.homeSearch)) {
              this.currentModule = undefined;
              this.isHomePage = true;
              this.ignoreGrid = true;
              return;
            }

            if (url.startsWith(urlSeparator + RoutesEnum.homePage)) {
              this.currentModule = RoutesEnum.homePage;
              this.isHomePage = true;
              this.ignoreGrid = true;
              return;
            }

            if (url.startsWith(urlSeparator + RoutesEnum.deposit)) {
              this.currentModule = RoutesEnum.deposit;
              return;
            }

            if (url.startsWith(urlSeparator + RoutesEnum.preservationSpace)) {
              this.currentModule = RoutesEnum.preservationSpace;
              return;
            }

            if (url.startsWith(urlSeparator + RoutesEnum.admin)) {
              this.currentModule = RoutesEnum.admin;
              return;
            }

            if (url.startsWith(urlSeparator + RoutesEnum.preservationPlanning)) {
              this.currentModule = RoutesEnum.preservationPlanning;
              return;
            }

            if (url.startsWith(urlSeparator + RoutesEnum.order)) {
              this.currentModule = RoutesEnum.order;
              return;
            }

            this.currentModule = null;
          }
        }),
      ),
    );
  }

  logout(): void {
    this._store.dispatch(new AppAction.Logout());
  }

  getTitle(): string {
    return this._document.title;
  }

  themeChange(theme: ThemeEnum): void {
    if (theme !== this._store.selectSnapshot((state: LocalStateModel) => state.application.theme)) {
      this._store.dispatch(new AppAction.ChangeAppTheme(theme));
    }
  }

  navigateToHome(): void {
    this.navigate(RoutesEnum.homePage);
  }

  private _observeThemeChange(): Observable<ThemeEnum> {
    return this.themeObs.pipe(
      distinctUntilChanged(),
      tap((theme: ThemeEnum) => {
        this.setTheme();
        this.setFavicon();
        this.setImageToolbar();
      }),
    );
  }

  private listenLanguageChange(): Observable<Enums.Language.LanguageEnum> {
    return this.appLanguageObs.pipe(
      tap((language: Enums.Language.LanguageEnum) => {
        this._setTitle();
      }),
    );
  }

  private listenAppReady(): Observable<boolean> {
    return this.isApplicationInitializedObs.pipe(
      tap((isInit: boolean) => {
        this._setTitle();
      }),
    );
  }

  private _setOverrideCssStyleSheet(): void {
    const header = this._document.querySelector("head");
    const link = this._document.createElement("link");
    this._renderer.setAttribute(link, "type", "text/css");
    this._renderer.setAttribute(link, "rel", "stylesheet");
    this._renderer.setAttribute(link, "href", `./assets/styles/override.css`);
    this._renderer.appendChild(header, link);
  }

  computeDisplayCart(isLogged: boolean, userRoles: Enums.UserApplicationRole.UserApplicationRoleEnum[]): boolean {
    this.displayCart = PermissionUtil.isUserHavePermission(isLogged, ApplicationRolePermissionEnum.userPermission, userRoles);
    return this.displayCart;
  }

  search(searchTerm: string): void {
    HomeHelper.navigateToSearch(this._store, searchTerm);
  }

  private _activeChemicalMoleculePreviewIfEnable(): void {
    if (environment.visualizationChemicalMoleculeMode === ChemicalMoleculeVisualizationEnum.disabled) {
      return;
    }
    const threeDimensionalLib = environment.visualizationChemicalMolecule3dLib;
    const twoDimensionalLibs = environment.visualizationChemicalMolecule2dLibs;
    const listScript = [];
    this._document.querySelectorAll(`script[${this._ATTRIBUTE_SRC_DIFFER_KEY}]`).forEach(script => {
      listScript.push(script);
    });

    if (environment.visualizationChemicalMoleculeMode === ChemicalMoleculeVisualizationEnum.threeDimensionalOnly
      || environment.visualizationChemicalMoleculeMode === ChemicalMoleculeVisualizationEnum.threeAndTwoDimensional) {
      const jsmolScript = this._getScriptWithValueEndWith(listScript, threeDimensionalLib);
      if (isNullOrUndefined(jsmolScript)) {
        this._loggingService.logWarning(`Unable to enable chemical molecule visualization, missing ${threeDimensionalLib}`);
        return;
      }
      this._activeScriptDiffer([jsmolScript]);
    }

    if (environment.visualizationChemicalMoleculeMode === ChemicalMoleculeVisualizationEnum.threeAndTwoDimensional) {
      const listScriptToTreat = [];
      twoDimensionalLibs.forEach(libNameNeeded => {
        const scriptLibNeeded = this._getScriptWithValueEndWith(listScript, libNameNeeded);
        if (isNullOrUndefined(scriptLibNeeded)) {
          this._loggingService.logWarning(`Unable to enable chemical molecule visualization 2D, missing ${libNameNeeded}`);
          return;
        }
        listScriptToTreat.push(scriptLibNeeded);
      });
      if (listScriptToTreat.length !== twoDimensionalLibs.length) {
        this._loggingService.logWarning(`Swith to chemical molecule visualization 3D`);
        environment.visualizationChemicalMoleculeMode = ChemicalMoleculeVisualizationEnum.threeDimensionalOnly;
        return;
      }
      // Need to differ lib JSME to avoid error at startup
      setTimeout(() => {
        this._activeScriptDiffer(listScriptToTreat);
      }, 1000);
    }
  }

  private _activeScriptDiffer(listScript: Element[]): void {
    listScript.forEach(script => {
      const value = this._getValueScriptDiffer(script);
      if (isNullOrUndefined(value)) {
        return;
      }
      this._renderer.setAttribute(script, this._ATTRIBUTE_SRC, value);
      this._renderer.removeAttribute(script, this._ATTRIBUTE_SRC_DIFFER_KEY);
    });
  }

  private _getValueScriptDiffer(script: Element): string | undefined {
    if (isNullOrUndefined(script) || isNullOrUndefined(script.attributes)) {
      return undefined;
    }
    const attribute = script.attributes[this._ATTRIBUTE_SRC_DIFFER_KEY];
    return isNullOrUndefined(attribute) ? undefined : attribute.value;
  }

  private _getScriptWithValueEndWith(listScript: Element[], threeDimensionalLib: string): Element | undefined {
    return listScript.find(script => {
      const value = this._getValueScriptDiffer(script);
      return isNotNullNorUndefined(value) && value.endsWith(threeDimensionalLib);
    });
  }

  openUserGuideSidebar(): void {
    this._store.dispatch(new AppAction.ChangeDisplaySidebarUserGuide(true));
  }

  scrollHandler($event: Event): void {
    this._scrollService.triggerScrollEventMain($event);
  }
}
