import {ChangeDetectionStrategy, ChangeDetectorRef, Component, Input,} from "@angular/core";
import {FormArray, FormBuilder, Validators,} from "@angular/forms";
import {AccessOrganizationalUnit, ApplicationRole, Person,} from "@app/generated-api";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {UserApplicationRoleEnum} from "@shared/enums/user-application-role.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {UserExtended} from "@shared/models/business/user-extended.model";
import {isNullOrUndefined, PropertyName, SolidifyValidator,} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-user-form",
  templateUrl: "./admin-user-form.html",
  styleUrls: ["../../../../../shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminUserForm extends SharedAbstractFormPresentational<UserExtended> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  applicationRolesNames: string[] = [...Object.values(UserApplicationRoleEnum)];

  @Input()
  listPersons: Person[];

  @Input()
  listOrganizationalUnit: AccessOrganizationalUnit[];

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef);
  }

  protected bindFormTo(user: UserExtended): void {
    this.form = this._fb.group({
      [this.formDefinition.externalUid]: [user.externalUid, [Validators.required, SolidifyValidator]],
      [this.formDefinition.targetedUid]: [user.targetedUid],
      [this.formDefinition.firstName]: [user.firstName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastName]: [user.lastName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.homeOrganization]: [user.homeOrganization, [Validators.required, SolidifyValidator]],
      [this.formDefinition.email]: [user.email, [Validators.required, SolidifyValidator]],
      [this.formDefinition.person]: isNullOrUndefined(user.person) ? [] : [user.person.resId],
      [this.formDefinition.accessToken]: [user.accessToken, [Validators.required, SolidifyValidator]],
      [this.formDefinition.refreshToken]: [user.refreshToken, [Validators.required, SolidifyValidator]],
      [this.formDefinition.applicationRoles]: [this.fillApplicationRolesSelection(this.applicationRolesNames, user.applicationRoles)],
    });
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.externalUid]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.targetedUid]: [""],
      [this.formDefinition.firstName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.homeOrganization]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.email]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.person]: [""],
      [this.formDefinition.accessToken]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.refreshToken]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.applicationRoles]: [this.mapToCheckboxArrayGroup(this.applicationRolesNames)],
    });
  }

  protected treatmentBeforeSubmit(user: UserExtended): UserExtended {
    const userRoles = this.form.controls.applicationRoles.value.controls as FormArray[];
    const selectedArray: string [] = [];
    userRoles.map((formCtrl) => {
      if (formCtrl.value.selected) {
        selectedArray.push(formCtrl.value.name);
      }
    });
    user.applicationRoles = [];
    selectedArray.forEach(role => {
      user.applicationRoles.push({resId: role});
    });
    //create a new person with the corresponding resId, if needed.
    const resId = user.person;
    if (resId === undefined) {
      user.person = null;
    } else if (resId !== null) {
      user.person = {resId: resId as string};
    }
    return user;
  }

  get getApplicationRolesNamesArray(): FormArray {
    return this.form.controls.applicationRoles.value as FormArray;
  }

  /**
   * Create a mapping of a string based dataset
   * to a form array suitable for a multi checkbox array selection.
   * this provides a more concise solution
   * as oppose to working with [true, false, false, true]
   */
  private mapToCheckboxArrayGroup(data: any[]): FormArray {
    return this._fb.array(data.map((i) =>
      this._fb.group({
        name: i,
        selected: false,
      })));
  }

  private fillApplicationRolesSelection(data: any[], selectedRoles: ApplicationRole[]): FormArray {
    return this._fb.array(data.map((i) =>
      this._fb.group({
        name: i,
        selected: selectedRoles.some(r => r.resId === i),
      })));
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() externalUid: string;
  @PropertyName() targetedUid: string;
  @PropertyName() firstName: string;
  @PropertyName() lastName: string;
  @PropertyName() homeOrganization: string;
  @PropertyName() email: string;
  @PropertyName() person: string;
  @PropertyName() accessToken: string;
  @PropertyName() refreshToken: string;
  @PropertyName() applicationRoles: string;
}
