import {OrganizationalUnitPreservationPolicy} from "@admin/models/organizational-unit-preservation-policy.model";
import {OrganizationalUnitSubmissionPolicy} from "@admin/models/organizational-unit-submission-policy.model";
import {AdminOrgUnitAction, adminOrgUnitActionNameSpace,} from "@admin/orgunit/stores/admin-organizational-unit.action";
import {adminOrgUnitPreservationPolicyActionNameSpace} from "@admin/orgunit/stores/preservation-policy/admin-organizational-unit-preservation-policy.action";
import {
  AdminOrganizationalUnitPreservationPolicyState,
  AdminOrganizationalUnitPreservationPolicyStateModel,
} from "@admin/orgunit/stores/preservation-policy/admin-organizational-unit-preservation-policy.state";
import {adminOrgUnitSubmissionPolicyActionNameSpace} from "@admin/orgunit/stores/submission-policy/admin-organizational-unit-submission-policy.action";
import {
  AdminOrganizationalUnitSubmissionPolicyState,
  AdminOrganizationalUnitSubmissionPolicyStateModel,
} from "@admin/orgunit/stores/submission-policy/admin-organizational-unit-submission-policy.state";
import {OrganizationalUnitExtended} from "@deposit/models/organizational-unit-extended.model";
import {DepositStateModel} from "@deposit/stores/deposit.state";
import {AdminResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {LocalModelAttributeEnum} from "@shared/enums/model-attribute.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SharedPreservationPolicyAction} from "@shared/stores/preservation-policy/shared-preservation-policy.action";
import {SharedSubmissionPolicyAction} from "@shared/stores/submission-policy/shared-submission-policy.action";
import {Action, Actions, ofActionCompleted, State, StateContext, Store,} from "@ngxs/store";

import _ from "lodash";
import {Observable} from "rxjs";
import {tap} from "rxjs/internal/operators/tap";
import {
  ActionSubActionCompletionsWrapper,
  ApiService,
  ChangeModeEnum,
  defaultResourceStateInitValue,
  isNotNullNorUndefined,
  ModelAttributeEnum,
  NotificationService,
  ObjectUtil,
  OverrideDefaultAction,
  Relation2TiersActionHelper,
  Relation2TiersNameSpace,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
  TRANSLATE,
} from "solidify-frontend";

export interface AdminOrganizationalUnitStateModel extends ResourceStateModel<OrganizationalUnitExtended> {
  admin_organizationalUnit_submissionPolicy?: AdminOrganizationalUnitSubmissionPolicyStateModel;
  admin_organizationalUnit_preservationPolicy?: AdminOrganizationalUnitPreservationPolicyStateModel;
}

@State<AdminOrganizationalUnitStateModel>({
  name: LocalStateEnum.admin_organizationalUnit,
  defaults: {
    ...defaultResourceStateInitValue,
  },
  children: [
    AdminOrganizationalUnitSubmissionPolicyState,
    AdminOrganizationalUnitPreservationPolicyState,
  ],
})
export class AdminOrganizationalUnitState extends ResourceState<OrganizationalUnitExtended> {

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: adminOrgUnitActionNameSpace,
      urlResource: AdminResourceApiEnum.organizationalUnits,
      routeRedirectUrlAfterSuccessAction: RoutesEnum.adminOrganizationalUnit,
      notificationResourceCreateTextToTranslate: TRANSLATE("admin.organizationalUnit.notification.resource.create"),
      notificationResourceDeleteTextToTranslate: TRANSLATE("admin.organizationalUnit.notification.resource.delete"),
      notificationResourceUpdateTextToTranslate: TRANSLATE("admin.organizationalUnit.notification.resource.update"),
    });
  }

  @Action(AdminOrgUnitAction.LoadResource)
  loadResource(ctx: StateContext<DepositStateModel>, action: AdminOrgUnitAction.LoadResource): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    ctx.dispatch([
      new SharedPreservationPolicyAction.GetAll(),
      new SharedSubmissionPolicyAction.GetAll(),
      new AdminOrgUnitAction.LoadResourceSuccess(action),
    ]);
  }

  @Action(AdminOrgUnitAction.LoadResourceSuccess)
  loadResourceSuccess(ctx: StateContext<DepositStateModel>, action: AdminOrgUnitAction.LoadResourceSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(AdminOrgUnitAction.LoadResourceFail)
  loadResourceFail(ctx: StateContext<DepositStateModel>, action: AdminOrgUnitAction.LoadResourceFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @OverrideDefaultAction()
  @Action(AdminOrgUnitAction.Create)
  create(ctx: StateContext<AdminOrganizationalUnitStateModel>, action: AdminOrgUnitAction.Create): Observable<OrganizationalUnitExtended> {
    const oldDefaultSubmissionPolicyId = undefined;
    const oldDefaultPreservationPolicyId = undefined;
    return super.internalCreate(ctx, action)
      .pipe(
        tap(model => {
          const parentResId = model.resId;
          this.updateSubResourceSubmissionAndPreservationPolicy(ChangeModeEnum.create, action, ctx, oldDefaultSubmissionPolicyId, oldDefaultPreservationPolicyId, parentResId);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(AdminOrgUnitAction.Update)
  update(ctx: StateContext<AdminOrganizationalUnitStateModel>, action: AdminOrgUnitAction.Update): Observable<OrganizationalUnitExtended> {
    const oldDefaultSubmissionPolicyId = ObjectUtil.getValue(ctx.getState().current.defaultSubmissionPolicy, ModelAttributeEnum.resId) as string;
    const oldDefaultPreservationPolicyId = ObjectUtil.getValue(ctx.getState().current.defaultPreservationPolicy, ModelAttributeEnum.resId) as string;
    return super.internalUpdate(ctx, action)
      .pipe(
        tap(model => {
          const parentResId = action.modelFormControlEvent.model.resId;

          this.updateSubResourceSubmissionAndPreservationPolicy(ChangeModeEnum.update, action, ctx, oldDefaultSubmissionPolicyId, oldDefaultPreservationPolicyId, parentResId);
        }),
      );
  }

  private updateSubResourceSubmissionAndPreservationPolicy(changeMode: ChangeModeEnum, action: AdminOrgUnitAction.Create | AdminOrgUnitAction.Update, ctx: StateContext<AdminOrganizationalUnitStateModel>, oldDefaultSubmissionPolicyId: string | undefined, oldDefaultPreservationPolicyId: string | undefined, parentResId: string): void {
    const listSubmissionPoliciesActionSubActionCompletionsWrapper = this.getListSubresourceUpdateActionSubActionCompletionsWrapper(
      changeMode,
      adminOrgUnitSubmissionPolicyActionNameSpace,
      action,
      ctx,
      oldDefaultSubmissionPolicyId,
      parentResId,
      LocalModelAttributeEnum.defaultSubmissionPolicy,
      LocalModelAttributeEnum.submissionPolicies,
      LocalStateEnum.admin_organizationalUnit_submissionPolicy);

    const listPreservationPoliciesActionSubActionCompletionsWrapper = this.getListSubresourceUpdateActionSubActionCompletionsWrapper(
      changeMode,
      adminOrgUnitPreservationPolicyActionNameSpace,
      action,
      ctx,
      oldDefaultPreservationPolicyId,
      parentResId,
      LocalModelAttributeEnum.defaultPreservationPolicy,
      LocalModelAttributeEnum.preservationPolicies,
      LocalStateEnum.admin_organizationalUnit_preservationPolicy);

    StoreUtil.updateSubResourceCustom(changeMode, ctx, adminOrgUnitActionNameSpace, [...listSubmissionPoliciesActionSubActionCompletionsWrapper, ...listPreservationPoliciesActionSubActionCompletionsWrapper]);
  }

  private getListSubresourceUpdateActionSubActionCompletionsWrapper(changeMode: ChangeModeEnum,
                                                                    nameSpace: Relation2TiersNameSpace,
                                                                    action: AdminOrgUnitAction.Create | AdminOrgUnitAction.Update,
                                                                    ctx: StateContext<AdminOrganizationalUnitStateModel>,
                                                                    oldDefaultId: string | undefined,
                                                                    parentResId: string,
                                                                    defaultFormKey: string,
                                                                    selectedFormKey: string,
                                                                    subStateKey: LocalStateEnum): ActionSubActionCompletionsWrapper[] {
    const newDefaultId = action.modelFormControlEvent.model[defaultFormKey].resId;
    const oldSelectedValues = changeMode === ChangeModeEnum.create ? [] : _.map(ctx.getState()[subStateKey].selected, LocalModelAttributeEnum.resId);
    const newSelectedValues = action.modelFormControlEvent.model[selectedFormKey];
    return this.getUpdateRelationActionsWrapper(nameSpace, oldDefaultId, newDefaultId, parentResId, oldSelectedValues, newSelectedValues);
  }

  private getUpdateRelationActionsWrapper(nameSpace: Relation2TiersNameSpace, oldDefaultResId: string | undefined, newDefaultResId: string, parentResId: string, oldSelectedValuesId: string[], newSelectedValuesId: string[]): ActionSubActionCompletionsWrapper[] {
    const listActionSubActionCompletionsWrapper: ActionSubActionCompletionsWrapper[] = [];

    listActionSubActionCompletionsWrapper.push({
      action: Relation2TiersActionHelper.update(nameSpace, parentResId, oldSelectedValuesId, newSelectedValuesId),
      subActionCompletions: [
        this.actions$.pipe(ofActionCompleted(nameSpace.UpdateSuccess)),
        this.actions$.pipe(ofActionCompleted(nameSpace.UpdateFail)),
      ],
    });

    if (oldDefaultResId !== newDefaultResId) {
      if (isNotNullNorUndefined(oldDefaultResId) && newSelectedValuesId.includes(oldDefaultResId)) {
        listActionSubActionCompletionsWrapper.push({
          action: Relation2TiersActionHelper.updateRelation(nameSpace, parentResId, oldDefaultResId, {defaultPolicy: false} as OrganizationalUnitSubmissionPolicy | OrganizationalUnitPreservationPolicy),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(nameSpace.UpdateRelationSuccess)),
            this.actions$.pipe(ofActionCompleted(nameSpace.UpdateRelationFail)),
          ],
        });
      }

      if (isNotNullNorUndefined(newDefaultResId)) {
        listActionSubActionCompletionsWrapper.push({
          action: Relation2TiersActionHelper.updateRelation(nameSpace, parentResId, newDefaultResId, {defaultPolicy: true} as OrganizationalUnitSubmissionPolicy | OrganizationalUnitPreservationPolicy),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(nameSpace.UpdateRelationSuccess)),
            this.actions$.pipe(ofActionCompleted(nameSpace.UpdateRelationFail)),
          ],
        });
      }
    }
    return listActionSubActionCompletionsWrapper;
  }
}
