import {ChangeDetectionStrategy, ChangeDetectorRef, Component, Input,} from "@angular/core";
import {FormBuilder, Validators,} from "@angular/forms";
import {OrganizationalUnitExtended} from "@deposit/models/organizational-unit-extended.model";
import {PreservationPolicy, SubmissionPolicy,} from "@app/generated-api";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {LocalModelAttributeEnum} from "@shared/enums/model-attribute.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import _ from "lodash";
import {DateUtil, PropertyName, SolidifyValidator,} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-orgunit-form",
  templateUrl: "./admin-orgunit-form.presentational.html",
  styleUrls: ["../../../../../shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminOrgunitFormPresentational extends SharedAbstractFormPresentational<OrganizationalUnitExtended> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  @Input()
  selectedSubmissionPolicies: SubmissionPolicy[];

  @Input()
  selectedPreservationPolicies: PreservationPolicy[];

  @Input()
  listSubmissionPolicies: SubmissionPolicy[];

  @Input()
  listPreservationPolicies: PreservationPolicy[];

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: ["", [SolidifyValidator]],
      [this.formDefinition.openingDate]: [""],
      [this.formDefinition.closingDate]: [""],
      [this.formDefinition.isEmpty]: [false],
      [this.formDefinition.open]: [false],
      [this.formDefinition.submissionPolicies]: [[], [Validators.required, SolidifyValidator]],
      [this.formDefinition.preservationPolicies]: [[], [Validators.required, SolidifyValidator]],
      [this.formDefinition.defaultSubmissionPolicy]: [undefined, [Validators.required, SolidifyValidator]],
      [this.formDefinition.defaultPreservationPolicy]: [undefined, [Validators.required, SolidifyValidator]],
      [this.formDefinition.url]: ["", [SolidifyValidator]],
    });
  }

  protected bindFormTo(organizationalUnit: OrganizationalUnitExtended): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: [organizationalUnit.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: [organizationalUnit.description, [SolidifyValidator]],
      [this.formDefinition.openingDate]: [organizationalUnit.openingDate],
      [this.formDefinition.closingDate]: [organizationalUnit.closingDate],
      [this.formDefinition.isEmpty]: [organizationalUnit.isEmpty],
      [this.formDefinition.open]: [organizationalUnit.open],
      [this.formDefinition.submissionPolicies]: [_.map(this.selectedSubmissionPolicies, LocalModelAttributeEnum.resId), [Validators.required, SolidifyValidator]],
      [this.formDefinition.preservationPolicies]: [_.map(this.selectedPreservationPolicies, LocalModelAttributeEnum.resId), [Validators.required, SolidifyValidator]],
      [this.formDefinition.defaultSubmissionPolicy]: [(organizationalUnit.defaultSubmissionPolicy ? organizationalUnit.defaultSubmissionPolicy.resId : undefined), [Validators.required, SolidifyValidator]],
      [this.formDefinition.defaultPreservationPolicy]: [(organizationalUnit.defaultPreservationPolicy ? organizationalUnit.defaultPreservationPolicy.resId : undefined), [Validators.required, SolidifyValidator]],
      [this.formDefinition.url]: [organizationalUnit.url, [SolidifyValidator]],

    });
  }

  protected treatmentBeforeSubmit(organizationalUnit: OrganizationalUnitExtended | any): OrganizationalUnitExtended {
    organizationalUnit.openingDate = DateUtil.convertToLocalDateDateSimple(organizationalUnit.openingDate);
    organizationalUnit.closingDate = DateUtil.convertToLocalDateDateSimple(organizationalUnit.closingDate);
    organizationalUnit.defaultSubmissionPolicy = {resId: organizationalUnit[this.formDefinition.defaultSubmissionPolicy]};
    organizationalUnit.defaultPreservationPolicy = {resId: organizationalUnit[this.formDefinition.defaultPreservationPolicy]};
    return organizationalUnit;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() name: string;
  @PropertyName() description: string;
  @PropertyName() isEmpty: string;
  @PropertyName() closingDate: string;
  @PropertyName() openingDate: string;
  @PropertyName() submissionPolicies: string;
  @PropertyName() preservationPolicies: string;
  @PropertyName() open: string;
  @PropertyName() defaultSubmissionPolicy: string;
  @PropertyName() defaultPreservationPolicy: string;
  @PropertyName() url: string;
}
