import {ChangeDetectionStrategy, ChangeDetectorRef, Component,} from "@angular/core";
import {FormArray, FormBuilder, Validators,} from "@angular/forms";
import {Oauth2Client} from "@app/generated-api";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {PropertyName, SolidifyValidator,} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-oauth-form",
  templateUrl: "./admin-oauth2-client-form.presentational.html",
  styleUrls: ["../../../../../shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminOAtuh2ClientFormPresentational extends SharedAbstractFormPresentational<Oauth2Client> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();
  public availableGrantTypes: any[] = [...Object.values(GrantTypesEnum)];
  hide: boolean = true;

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef);
  }

  protected bindFormTo(oauth2Client: Oauth2Client): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: [oauth2Client.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.clientId]: [oauth2Client.clientId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.secret]: [oauth2Client.secret, [Validators.required, SolidifyValidator]],
      [this.formDefinition.redirectUri]: [oauth2Client.redirectUri, [Validators.required, SolidifyValidator]],
      [this.formDefinition.scope]: [oauth2Client.scope, [Validators.required, SolidifyValidator]],
      [this.formDefinition.grantTypes]: [this.prefillGrantSelection(this.availableGrantTypes, oauth2Client.grantTypes)],
      [this.formDefinition.accessTokenValiditySeconds]: [oauth2Client.accessTokenValiditySeconds, [Validators.required, SolidifyValidator]],
      [this.formDefinition.refreshTokenValiditySeconds]: [oauth2Client.refreshTokenValiditySeconds, [Validators.required, SolidifyValidator]],
    });
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.clientId]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.secret]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.redirectUri]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.scope]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.grantTypes]: [this.mapToCheckboxArrayGroup(this.availableGrantTypes)],
      [this.formDefinition.accessTokenValiditySeconds]: [84600, [Validators.required, SolidifyValidator]],
      [this.formDefinition.refreshTokenValiditySeconds]: [84600, [Validators.required, SolidifyValidator]],
    });
  }

  get grantTypesArray(): FormArray {
    return this.form.controls.grantTypes.value as FormArray;
  }

  protected treatmentBeforeSubmit(oauth2Client: Oauth2Client): Oauth2Client {
    const grantTypes = this.form.controls.grantTypes.value.controls as FormArray [];
    const selectedArray: string [] = [];
    grantTypes.map((formCtrl) => {
      if (formCtrl.value.selected) {
        selectedArray.push(formCtrl.value.name);
      }
    });
    oauth2Client.grantTypes = selectedArray;
    return oauth2Client;
  }

  /**
   * Create a mapping of a string based dataset
   * to a form array suitable for a multi checkbox array selection.
   * this provides a more concise solution
   * as oppose to working with [true, false, false, true]
   */
  private mapToCheckboxArrayGroup(data: any[]): FormArray {
    return this._fb.array(data.map((i) =>
      this._fb.group({
        name: i,
        selected: false,
      })));
  }

  private prefillGrantSelection(data: any[], selectedGrantTypes: any[]): FormArray {
    return this._fb.array(data.map((i) =>
      this._fb.group({
        name: i,
        selected: selectedGrantTypes.includes(i),
      })));
  }

}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() name: string;
  @PropertyName() clientId: string;
  @PropertyName() secret: string;
  @PropertyName() redirectUri: string;
  @PropertyName() scope: string;
  @PropertyName() grantTypes: string;
  @PropertyName() accessTokenValiditySeconds: string;
  @PropertyName() refreshTokenValiditySeconds: string;
}

export enum GrantTypesEnum {
  IMPLICIT = "implicit",
  AUTH_CODE = "authorization_code",
  REFRESH_TOKEN = "refresh_token",
  CLIENT_CREDENTIALS = "client_credentials"
}
